#if defined DEFERRED || defined COMPOSITE
    struct gbuffers{
        vec4 spec;
        vec4 albedo;
        vec3 normal;
        vec3 view_normal;
        vec2 depth;
        vec2 lightmap;
        float id;
    };

    gbuffers gbuffers_data(vec2 coord){
        gbuffers g;

        uvec4 gbuffers9 = texture(colortex9, coord);

        g.albedo = unpackUnorm4x8(gbuffers9.x);
        g.spec = unpackUnorm4x8(gbuffers9.y);

        g.albedo.xyz = srgb_to_linear(g.albedo.xyz + texture2(colortex14, coord).xyz);
        g.normal = decode_unit_vector(unpackSnorm4x8(gbuffers9.z).xy);
        g.view_normal = mat3(gbufferModelView) * g.normal;

        g.depth.x = texture2(depthtex0, coord).x;
        g.depth.y = texture2(depthtex1, coord).x;

        g.lightmap = unpackUnorm4x8(gbuffers9.w).yz;

        g.id = unpackUnorm4x8(gbuffers9.w).x * 255.0;

        return g;
    }

    struct specular{
        float f0;

        float roughness;
        float porosity;
        float emissive;
        float ior;

        vec3 k;
        vec3 n;
    };

    specular specular_data(vec4 spec, vec4 albedo, float ID){
        specular s;

        s.roughness = 1.0 - spec.x;
        s.f0 = spec.y;
        s.porosity = spec.z;
        s.emissive = spec.w;
        s.ior = 1.0;

        if(ID == 2){
            //s.roughness = 0.021;
            s.f0 = 0.20;
            s.ior = 1.33283;
        }

        if(ID == 3){
            s.roughness = 0.018;
            s.f0 = 0.22;
            s.ior = 1.309;
        }

        if(ID == 4){
            #ifdef STAINED_GLASS_REFLECTION
                s.roughness = 0.042;
                s.f0 = 0.52;
            #endif
            s.ior = 1.51714;
        }

        if(ID == 5){
            s.roughness = 0.036;
            s.f0 = 0.44;
            s.ior = 1.484;
        }

        if(s.f0 == 230){
            s.n = vec3(2.9114, 2.9497, 2.5845);
            s.k = vec3(0.18299, 0.42108, 1.3734);
        } else if(s.f0 == 231){
            s.n = vec3(0.18299, 0.42108, 1.3734);
            s.k = vec3(3.4242, 2.3459, 1.7704);
        } else if(s.f0 == 231){
            s.n = vec3(1.3456, 0.96521, 0.61722);
            s.k = vec3(7.4746, 6.3995, 5.3031);
        } else if(s.f0 == 231){
            s.n = vec3(3.1071, 3.1812, 2.3230);
            s.k = vec3(3.3314, 3.3291, 3.1350);
        } else if(s.f0 == 231){
            s.n = vec3(0.27105, 0.67693, 1.3164);
            s.k = vec3(3.6092, 2.6248, 2.2921);
        } else if(s.f0 == 231){
            s.n = vec3(1.9100, 1.8300, 1.4400);
            s.k = vec3(3.5100, 3.4000, 3.1800);
        } else if(s.f0 == 231){
            s.n = vec3(2.3757, 2.0847, 1.8453);
            s.k = vec3(4.2655, 3.7153, 3.1365);
        } else if(s.f0 == 231){
            s.n = vec3(0.15943, 0.14512, 0.13547);
            s.k = vec3(3.9291, 3.1900, 2.3808);
        }

        return s;
    }

    struct space_position{
        vec3 view_position0;
        vec3 view_position1;

        vec3 position0;
        vec3 position1;

        vec3 vector0;
        vec3 vector1;

        vec3 inverse_position0;
        vec3 inverse_position1;

        vec3 inverse_vector0;
        vec3 inverse_vector1;

        vec3 sun_vector_view;
        vec3 moon_vector_view;
        vec3 light_vector_view;

        vec3 sun_vector;
        vec3 moon_vector;
        vec3 light_vector;
    };

    space_position position(vec2 coord, vec2 depth, bool temporal){
        space_position sp;

        if(temporal) coord = coord - jitter * 0.5;

        sp.view_position0 = vec3(coord, depth.x);
        sp.view_position1 = vec3(coord, depth.y);

        sp.position0 = screen_to_viewspace(sp.view_position0);
        sp.position1 = screen_to_viewspace(sp.view_position1);
		//Strasbourg
        sp.inverse_position0 = view_to_worldspace(sp.position0);
        sp.inverse_position1 = view_to_worldspace(sp.position1);

        sp.vector0 = normalize(sp.position0);
        sp.vector1 = normalize(sp.position1);

        sp.inverse_vector0 = normalize(sp.inverse_position0);
        sp.inverse_vector1 = normalize(sp.inverse_position1);

        #if defined OVER_WORLD
            sp.sun_vector_view = normalize(sunPosition);
            sp.moon_vector_view = normalize(moonPosition);
            sp.light_vector_view = normalize(shadowLightPosition);
        #else
            vec3 view_position = mat3(shadowModelViewInverse) * vec3(0.0, 0.0, 1.0); 

            sp.sun_vector_view = normalize(mat3(gbufferModelView) * view_position);
            sp.moon_vector_view = sp.sun_vector_view;
            sp.light_vector_view = sp.sun_vector_view;
        #endif

        sp.sun_vector = mat3(gbufferModelViewInverse) * sp.sun_vector_view;
        sp.moon_vector = mat3(gbufferModelViewInverse) * sp.moon_vector_view;
        sp.light_vector = mat3(gbufferModelViewInverse) * sp.light_vector_view;

        return sp;
    }

    struct materials{
        bool atmosphere;
        bool translucent;

        bool water; 
        bool ice;
        bool stained_glass;
        bool slime;

        bool grass;
        bool leaves;

        bool torch;
        bool glowstone;
        bool lava;
        bool fire;
        bool redstone;

        bool textured;
        bool hand;
        bool entity;
        bool enchant;
        bool basic;
    };
    //Weiler
    materials material(vec2 depth, float ID){
        materials m;

        m.atmosphere = depth.y >= 1.0;
        m.translucent = depth.y > depth.x;

        m.water = condition(ID, 2);
        m.ice = condition(ID, 3);
        m.stained_glass = condition(ID, 4);
        m.slime = condition(ID, 5);

        m.grass = condition(ID, 6);
        m.leaves = condition(ID, 7);

        m.torch = condition(ID, 8);
        m.glowstone = condition(ID, 9);
        m.lava = condition(ID, 10);
        m.fire = condition(ID, 11);
        m.redstone = condition(ID, 12);

        m.textured = condition(ID, 100);
        m.entity = condition(ID, 101);
        m.hand = condition(ID, 102);
        m.enchant = condition(ID, 103);
        m.basic = condition(ID, 104);

        /*
        m.ior = 1.0002926;
        m.ior = m.water || isEyeInWater == 1 ? 1.33283 : m.ior;
        m.ior = m.ice ? 1.309 : m.ior;
        m.ior = m.stained_glass ? 1.51714 : m.ior;
        m.ior = m.slime ? 1.484 : m.ior;

        m.metal = 1.0;
        m.metal = ID == 201 ? 2 : m.metal;         //Fer
        m.metal = ID == 202 ? 3 : m.metal;         //Or
        m.metal = ID == 203 ? 4 : m.metal;         //Aluminium
        m.metal = ID == 204 ? 5 : m.metal;         //Chrome
        m.metal = ID == 205 ? 6 : m.metal;         //Cuivre (ou Couivre si vous êtes Aurelien_sama)
        m.metal = ID == 206 ? 7 : m.metal;         //Plomb
        m.metal = ID == 207 ? 8 : m.metal;         //Platine
        m.metal = ID == 208 ? 9 : m.metal;         //Silver
        */

        return m;
    }
#endif

struct atmosphere_constant{
    int steps0;
    int steps1;

    vec3 lambda_rayleigh;
    vec3 lambda_mie;
    vec3 lambda_ozone;

    vec3 sun_luminance, sun_illuminance;
    vec3 moon_luminance, moon_illuminance;

    float sun_radius, sun_distance, sun_angular_radius;
    float moon_radius, moon_distance, moon_angular_radius;

    float rayleigh_height;
    float mie_height;
    float mie_g;

    float planet_radius;
    float atmosphere_radius;

    mat2x3 scattering_coefficient;
    mat3 attenuation_coefficient;
};

atmosphere_constant atmosphere_s(){
    atmosphere_constant ac;

    ac.steps0 = 16;
    ac.steps1 = 8;

    ac.lambda_rayleigh = vec3(6.29673e-6, 1.21427e-5, 2.86453e-5);
    ac.lambda_mie = vec3(4e-7);
    ac.lambda_ozone = vec3(1.36820899679147, 3.31405330400124, 0.13601728252538) * 0.6e-6 * 2.5035422;

    ac.rayleigh_height = 8.5e3;
    ac.mie_height = 1.2e3;
    ac.mie_g = 0.76;

    ac.scattering_coefficient = mat2x3(ac.lambda_rayleigh, ac.lambda_mie);
    ac.attenuation_coefficient = mat3(ac.lambda_rayleigh, ac.lambda_mie * 1.11, ac.lambda_ozone);

    #if ATMOSPHERE_TYPE == 0
        ac.planet_radius = 6371e3;
        ac.atmosphere_radius = 6471e3;
        ac.sun_radius = 696340e3;
        ac.sun_distance = 149597870.7e3;
    #elif ATMOSPHERE_TYPE == 1
        ac.planet_radius = 1737e3;
        ac.atmosphere_radius = 1738e3;
        ac.sun_radius = 696340e3;
        ac.sun_distance = 152503397e3;
    #endif

    ac.moon_radius = 1737e3;
    ac.moon_distance = 38440e4;
    //Wissembourg
    ac.sun_angular_radius = 2.0 * ac.sun_radius / ac.sun_distance;
    ac.moon_angular_radius = 2.0 * ac.moon_radius / ac.moon_distance;

    ac.sun_illuminance = vec3(tau * (1.0 - cos(ac.sun_angular_radius)));
    ac.moon_illuminance = vec3(tau * (1.0 - cos(ac.moon_angular_radius)));

    ac.sun_luminance = vec3(125e3) * pi * (1.0 + SUNLIGHT_BRIGHTNESS);
    ac.moon_luminance = 0.318 * (ac.sun_luminance * rpi) * ac.moon_illuminance * (1.0 + MOONLIGHT_BRIGHTNESS);
        
    return ac;
}

struct volumetric_light_settings{
    int steps;

    float altitude;
    float density;
    float mie_g;
    float treshold;
};

volumetric_light_settings volumetric_light_s(){
    volumetric_light_settings vl;

    vl.steps = VOLUMETRIC_LIGHT_QUALITY;

    vl.altitude = 64.0;
    vl.density = 24.0 * VOLUMETRIC_LIGHT_DENSITY;
    vl.mie_g = 0.76;
    vl.treshold = 0.1;

    return vl;
}

struct volumetric_fog_settings{
    int octaves0;
    int octaves1;

    int directSteps;
    int indirectSteps;

    int steps0;
    int steps1;
    
    float altitude_min;
    float altitude_max;

    float thickness;
    float density;
    float speed;
    float coverage;
    float scale;
    float treshold;

    float weatherCoverage;
};

volumetric_fog_settings s_volume_fog(){
    volumetric_fog_settings vf;

    vf.octaves0 = 2;
    vf.octaves1 = 3;

    vf.steps0 = VOLUMETRIC_FOG_QUALITY;
    vf.steps1 = 3;

    vf.directSteps = 4;
    vf.indirectSteps = 2;

    vf.scale = 200.0;
    vf.thickness = 64.0 * VOLUMETRIC_FOG_THICKNESS;
    vf.altitude_min = VOLUMETRIC_FOG_ALTITUDE;
    vf.altitude_max = vf.altitude_min + vf.thickness;

    vf.coverage = 0.4 * VOLUMETRIC_FOG_COVERAGE;
    vf.weatherCoverage = 0.85;

    vf.density = 0.125 * VOLUMETRIC_FOG_DENSITY;
    vf.speed = 0.3 * VOLUMETRIC_FOG_SPEED;

    vf.treshold = 0.1;

    return vf;
}

struct clouds_3D_settings{
    int steps0;
    int steps1;

    int octaves0;
    int octaves1;

    float altitude_min;
    float altitude_max;

    float scale_map;
    float scale_mask;
    float scale_detail;

    float coverage;
    float weather_coverage;

    float thickness;
    float density;
    float time_shift;
    float treshold;

    float resolution;
    float distance;
};

clouds_3D_settings s_clouds3D(){
    clouds_3D_settings vc;

    vc.steps0 = VOLUMETRIC_CLOUD_QUALITY;
    vc.steps1 = 8;

    vc.octaves0 = 3;
    vc.octaves1 = 4;

    vc.scale_mask = 2.5e4;
    vc.scale_detail = 500.0;
    
    vc.coverage = 0.8 / VOLUMETRIC_CLOUD_COVERAGE;
    vc.weather_coverage = 0.02 / VOLUMETRIC_CLOUD_WEATHER_COVERAGE;

    vc.thickness = 1800.0 * VOLUMETRIC_CLOUD_THICKNESS;
    vc.density = (0.125 * VOLUMETRIC_CLOUD_DENSITY) * 700.0 / vc.thickness;
    vc.time_shift = 8.0 * VOLUMETRIC_CLOUD_SPEED;
    vc.treshold = 0.05;

    vc.altitude_min = VOLUMETRIC_CLOUD_ALTITUDE;
    vc.altitude_max = vc.altitude_min + vc.thickness;

    //Cloud shadow part
    vc.resolution = 1024.0;
    vc.distance = 500.0;

    return vc;
}

struct clouds_2D_settings{
    int octaves;

    float altitude_min;
    float altitude_max;

    float scale_mask;
    float scale_detail;

    float coverage;
    float weather_coverage;

    float thickness;
    float density;
    float time_shift;
    float treshold;
};

clouds_2D_settings s_clouds2D(){
    clouds_2D_settings pc;

    pc.octaves = 3;

    pc.scale_mask = 2e4;
    pc.scale_detail = 3400.0;

    pc.coverage = 0.85 * PLANAR_CLOUD_COVERAGE;
    pc.weather_coverage = 1.4;

    pc.thickness = 1400.0;
    pc.density = (0.05 * PLANAR_CLOUD_DENSITY) * 4.0 / pc.thickness;
    pc.time_shift = 20.0 * PLANAR_CLOUD_SPEED;
    pc.treshold = 0.1;

    pc.altitude_min = PLANAR_CLOUD_ALTITUDE;
    pc.altitude_max = pc.altitude_min + pc.thickness;

    return pc;
}

struct shadow_settings{
    int steps0;
    int steps1;
    int steps2;
    int steps3;

    float resolution;
    float depth;
    float distortion;

    float brightness;
    float radius;

    float z_depth;
    float strength;
};

shadow_settings shadow_s(){
    shadow_settings s;

    s.steps0 = 12;
    s.steps1 = 8;

    s.resolution = shadowMapResolution * MC_SHADOW_QUALITY;
    s.depth = 0.25;
    s.distortion = 0.9;

    //RSM part
    s.steps2 = GI_QUALITY;
    s.brightness = 1.0 + GI_BRIGHTNESS;
    s.radius = GI_RADIUS;

    //Screenspace contact shadow part
    s.steps3 = 8;
    s.z_depth = 0.05;
    s.strength = 1.0;
    
    return s;
}

struct water_settings{
    int steps;

    float mie_g;
    float density;
    float treshold;
        
    vec3 scattering_coefficient;
    vec3 attenuation_coefficient;
};

water_settings water_s(){
    water_settings ws;

    ws.steps = 8;
        
    ws.mie_g = 0.76;
    ws.treshold = 0.1;
        
    #if WATER_COLOR_PRESET == 0
        ws.density = 0.2 * WATER_FOG_DENSITY;

        ws.scattering_coefficient.r = 0.34 + WATER_SCATTERING_R;
        ws.scattering_coefficient.g = 0.82 + WATER_SCATTERING_G;
        ws.scattering_coefficient.b = 1.00 + WATER_SCATTERING_B;
        ws.scattering_coefficient *=  ws.density;

        ws.attenuation_coefficient.r = 0.10 + WATER_ATTENUATION_R;
        ws.attenuation_coefficient.g = 0.41 + WATER_ATTENUATION_G;
        ws.attenuation_coefficient.b = 0.58 + WATER_ATTENUATION_B;
    #elif WATER_COLOR_PRESET == 1
        ws.density = 0.3 * WATER_FOG_DENSITY;

        ws.scattering_coefficient.r = 0.06 + WATER_SCATTERING_R;
        ws.scattering_coefficient.g = 0.66 + WATER_SCATTERING_G;
        ws.scattering_coefficient.b = 0.69 + WATER_SCATTERING_B;
        ws.scattering_coefficient *= ws.density;

        ws.attenuation_coefficient.r = 0.23 + WATER_ATTENUATION_R;
        ws.attenuation_coefficient.g = 0.52 + WATER_ATTENUATION_G;
        ws.attenuation_coefficient.b = 0.55 + WATER_ATTENUATION_B;
    #elif WATER_COLOR_PRESET == 2
        ws.density = 0.75 * WATER_FOG_DENSITY;

        ws.scattering_coefficient.r = 0.2 + WATER_SCATTERING_R;
        ws.scattering_coefficient.g = 0.0 + WATER_SCATTERING_G;
        ws.scattering_coefficient.b = 0.0 + WATER_SCATTERING_B;
        ws.scattering_coefficient *= ws.density;

        ws.attenuation_coefficient.r = 0.4 + WATER_ATTENUATION_R;
        ws.attenuation_coefficient.g = 0.0 + WATER_ATTENUATION_G;
        ws.attenuation_coefficient.b = 0.0 + WATER_ATTENUATION_B;
    #elif WATER_COLOR_PRESET == 3
        ws.density = 0.5 * WATER_FOG_DENSITY;

        ws.scattering_coefficient.r = 0.40 + WATER_SCATTERING_R;
        ws.scattering_coefficient.g = 0.50 + WATER_SCATTERING_G;
        ws.scattering_coefficient.b = 0.05 + WATER_SCATTERING_B;
        ws.scattering_coefficient *= ws.density;
        
        ws.attenuation_coefficient.r = 0.30 + WATER_ATTENUATION_R;
        ws.attenuation_coefficient.g = 0.40 + WATER_ATTENUATION_G;
        ws.attenuation_coefficient.b = 0.10 + WATER_ATTENUATION_B;

        ws.scattering_coefficient *= 0.5;
    #endif

    ws.attenuation_coefficient = -log2(ws.attenuation_coefficient);

    return ws;
}