vec3 volumetric_light_density(atmosphere_constant ac, volumetric_light_settings vl, vec3 position){
    float position0 = position.y - vl.altitude;
    float position1 = length(position) - vl.altitude;

    float r = exp2(-position0 / ac.rayleigh_height) * mix(1.0, 32.0, wetness);

    float m = exp2(-position0 / ac.mie_height);
          m += exp2(-position0 * 0.025) * 12.0;

    return vec3(r, m, 0.0) * vl.density;
}

vec3 volumetric_light_transmittance(atmosphere_constant ac, volumetric_light_settings vl, vec3 position, vec3 direction, vec3 shadow, const int steps){
    float ray_length = (25.0 / steps) / abs(direction.y);

    vec3 ray_increment = direction * ray_length;
    vec3 ray_position = position + ray_increment * 0.5;

    if(any(lessThanEqual(shadow, vec3(0.0)))) return vec3(1.0);
 
    vec3 transmittance = vec3(0.0);

    for(int i = 0; i < steps; ++i, ray_position += ray_increment){
        transmittance += volumetric_light_density(ac, vl, ray_position);
    }

    return exp(-ac.attenuation_coefficient * transmittance * ray_length);
}

vec4 volumetric_light(vec3 position0, vec3 position1, vec3 vector, vec3 direction, vec3 sun_direction, vec3 moon_direction, vec3 noise, bool atmosphere){
    atmosphere_constant ac = atmosphere_s();
    volumetric_light_settings vl = volumetric_light_s();

    float VdotL = dot(vector, direction);
    float phase0 = phase_rayleigh(VdotL);
    float phase1 = phase_mie(VdotL, ac.mie_g);

    vec3 up_direction = vec3(0.0, 1.0, 0.0);

    if(atmosphere){
        position1 = position0 + vector * distance(position0, position1);
    }

    vec3 ray_increment = (position1 - position0) / float(vl.steps);
    vec3 ray_position = position0 + ray_increment * noise.x; ray_position += cameraPosition;

    vec3 shadow_position0 = world_to_shadowspace(position0);
    vec3 shadow_position1 = world_to_shadowspace(position1);

    vec3 shadow_increment = (shadow_position1 - shadow_position0) / float(vl.steps);
    vec3 shadow_position = shadow_position0 + shadow_increment * noise.x;

    float ray_length = length(ray_increment);

    vec3 direct_scattering = vec3(0.0), indirect_scattering = vec3(0.0), transmittance = vec3(1.0);

    for(int i = 0; i < vl.steps; ++i, ray_position += ray_increment, shadow_position += shadow_increment){
        vec3 density = volumetric_light_density(ac, vl, ray_position) * ray_length;
        if(density.y > 1e35) break;

        vec3 optical_depth = ac.attenuation_coefficient * density;
        vec3 step_transmittance = exp2(-optical_depth);
        vec3 step_transmittance_fraction = saturate((step_transmittance - 1.0) / -optical_depth);

        vec3 shadow = volumetric_shadow(shadow_position, ray_position - cameraPosition, false);

        vec3 o_transmittance = transmittance * step_transmittance_fraction;
        vec3 s_transmittance = o_transmittance * volumetric_light_transmittance(ac, vl, ray_position, direction, shadow, 3);
        vec3 v_transmittance = o_transmittance * volumetric_light_transmittance(ac, vl, ray_position, up_direction, shadow, 3);

        direct_scattering += ac.scattering_coefficient * (density.xy * vec2(phase0, phase1)) * s_transmittance * shadow;
        indirect_scattering += ac.scattering_coefficient * (density.xy * 0.25 * rpi) * v_transmittance;

        transmittance *= step_transmittance;
    }

    vec3 sunlight = atmospheric_transmittance(sun_direction, moon_direction);

    vec3 skylight = texture2(colortex1, project_sphere(up_direction)).xyz * 4.0;
         //skylight = mix(skylight, vec3(luminance(skylight)), wetness);
    
    float occlusion = mix(0.0, 1.0, pow8(eyeBrightnessSmooth.y / 240.0));

    vec3 scattering = direct_scattering * sunlight * (1.0 - wetness);
         scattering += indirect_scattering * skylight * occlusion;
         scattering = mix(scattering, vec3(luminance(scattering)), wetness);
    
    #if ATMOSPHERE_TYPE == 0
        return vec4(scattering, luminance(transmittance));
    #elif ATMOSPHERE_TYPE == 1
        return vec4(0.0, 0.0, 0.0, 1.0);
    #endif
}