float volumetric_fog_density(atmosphere_constant ac, volumetric_fog_settings vf, vec3 position){
    float animation = frameTimeCounter * vf.speed * pow2(rpi);
    float altitude = (position.y - vf.altitude_min) / vf.thickness;
    float attenuation = remap(altitude, 0.0, 0.2, 0.0, 1.0) * remap(altitude, 0.9, 1.0, 1.0, 0.0);

    vec3 direction0 = animation * normalize(vec3(-1.0, 0.0, 1.0));
    vec3 direction1 = animation * normalize(vec3(-1.0, 0.5, 1.0));

    vec3 position0 = position * 0.06 + direction0;
    vec3 position1 = position * 0.2 + direction1;

    float noise = perlin_fbm(position0, 4);

    float result = noise * attenuation * vf.coverage - (2.0 * attenuation * altitude * 0.5 + 0.5);
          result *= exp2(-max0(position.y - vf.altitude_min) * 0.35);

    return saturate(result) * vf.density;
}

float volumetric_fog_transmittance(atmosphere_constant ac, volumetric_fog_settings vf, vec3 position, vec3 direction, const int steps){
    float ray_length = 22.0;
    float transmittance = 0.0;

    for(int i = 0; i < steps; ++i, position += direction * ray_length){
        transmittance += ray_length * volumetric_fog_density(ac, vf, position);
    }

    return transmittance;
}

float volumetric_fog_phase(float c, float t){
    const float g0 = 0.35;
    const float g1 = 0.25;

    float phase0 = hg_phase(c,  pow(g0, t + 1.0));
    float phase1 = hg_phase(c, -pow(g0, t + 1.0));

    return mix(phase0, phase1, g1);
}

vec4 volumetric_fog(vec3 position0, vec3 position1, vec3 vector, vec3 direction, vec3 sun_direction, vec3 moon_direction, vec3 noise, bool atmosphere){
    atmosphere_constant ac = atmosphere_s();
    volumetric_fog_settings vf = s_volume_fog();

    #ifdef VOLUMETRIC_FOG_MORNING
        float sunrise = saturate(dot(sun_direction, vec3(1.0, 0.0, 0.0)));
              sunrise = pow4(sunrise * 0.7);

        if(sunrise <= 0.0){
            return vec4(0.0, 0.0, 0.0, 1.0);
        }
    #else
        float sunrise = 1.0;
    #endif

    vec3 up_direction = vec3(0.0, 1.0, 0.0);

    float VdotL = dot(normalize(position1), direction);
    float VdotU = dot(normalize(position1), up_direction);

    vec3 ray_increment = (position1 - position0) / float(vf.steps0);
    vec3 ray_position = position0 + ray_increment * noise.x; ray_position += cameraPosition;

    vec3 shadow_position0 = world_to_shadowspace(position0);
    vec3 shadow_position1 = world_to_shadowspace(position1);

    vec3 shadow_increment = (shadow_position1 - shadow_position0) / float(vf.steps0);
    vec3 shadow_position = shadow_position0 + shadow_increment * noise.x;

    float ray_length = length(ray_increment);

    vec3 direct_scattering = vec3(0.0), indirect_scattering = vec3(0.0);
    float transmittance = 1.0;

    for(int i = 0; i < vf.steps0; ++i, ray_position += ray_increment, shadow_position += shadow_increment){
        if(transmittance <= vf.treshold) break;

        float density = volumetric_fog_density(ac, vf, ray_position) * ray_length * sunrise;
        if(density <= 0.0) continue;

        float step_transmittance = exp2(-density);
        float step_transmittance_fraction = 1.0 - step_transmittance;

        vec3 shadow = volumetric_shadow(shadow_position, ray_position - cameraPosition, false);

        float direct_transmittance = volumetric_fog_transmittance(ac, vf, ray_position, direction, 12) * sunrise;
        float indirect_transmittance = volumetric_fog_transmittance(ac, vf, ray_position, up_direction, 6) * 0.1 * sunrise;

        for(int j = 0; j < vf.steps1; ++j){
            float a = pow(0.6, j);
            float b = pow(0.4, j);
            float c = pow(0.8, j);

            float s_transmittance = transmittance * (a * step_transmittance_fraction);

            float direct_step_transmittance = s_transmittance * exp(-b * direct_transmittance);
            float indirect_step_transmittance = s_transmittance * exp(-b * indirect_transmittance);

            float phase0 = volumetric_fog_phase(VdotL * c, direct_transmittance);
            float phase1 = 2.0 * volumetric_fog_phase(VdotU * c, indirect_transmittance);

            direct_scattering += direct_step_transmittance * phase0 * shadow;
            indirect_scattering += indirect_step_transmittance * phase1;
        }

        transmittance *= step_transmittance;
    }

    transmittance = saturate((transmittance - vf.treshold) / (1.0 - vf.treshold));

    vec3 sunlight = atmospheric_transmittance(sun_direction, moon_direction);

    vec3 skylight = texture2(colortex1, project_sphere(up_direction)).xyz * pi;
         //skylight = mix(skylight, vec3(luminance(skylight)), wetness);

    vec3 scattering = direct_scattering * sunlight;
         scattering += indirect_scattering * skylight;
         scattering = mix(scattering, vec3(luminance(scattering)), wetness);
    
    #if ATMOSPHERE_TYPE == 0
        return vec4(scattering, transmittance);
    #elif ATMOSPHERE_TYPE == 1
        return vec4(0.0, 0.0, 0.0, 1.0);
    #endif
}