float planar_clouds_density(clouds_2D_settings pc, vec3 position){
    const float offset = 1.0 / pc.scale_mask;
    const float scale = 1.0 / (24.0 * pc.scale_mask);

    float altitude = (position.y - pc.altitude_min) / pc.thickness;
    float animation = frameTimeCounter * pc.time_shift;

    vec3 direction0 = normalize(vec3(-1.0, 0.0, 1.0));
    vec3 direction1 = normalize(vec3(-1.0, 0.5, 1.0));

    float n0 = perlin_fbm(vec3(position.x + offset, position.y, position.z) * scale, pc.octaves);
    float n1 = perlin_fbm(vec3(position.x, position.y + offset, position.z) * scale, pc.octaves);
    float n2 = perlin_fbm(vec3(position.x, position.y, position.z + offset) * scale, pc.octaves);

    position = vec3(position.x * n0, position.y * n1, position.z * n2);
    
    vec3 position0 = (position + animation * direction0) / pc.scale_mask;
    vec3 position1 = (position + animation * direction1) / pc.scale_detail;

    float noise0 = perlin_fbm(position0, pc.octaves);
          noise0 = noise0 * pc.coverage * 2.0 - 1.0;

    if(noise0 <= 0.0) return 0.0;

    float noise1 = perlin_fbm(position1, pc.octaves) * 0.5;

    float result = 2.0 * saturate(noise0 - noise1);
          result = pow4(result);

    return result * pc.density;
}  

float planar_clouds_transmittance(clouds_2D_settings pc, vec3 position, vec3 direction, const int steps){
    float ray_length = 12.0;
    float transmittance = 0.0;

    for(int i = 0; i < steps; i++, position += direction * ray_length){
        transmittance += ray_length * planar_clouds_density(pc, position);
        if(transmittance <= 0.0) continue;

        //ray_length *= 1.5;
    }

    return transmittance;
}

vec4 planar_clouds(vec3 position, vec3 direction, vec3 sun_direction, vec3 moon_direction, vec3 atmosphere){
    atmosphere_constant ac = atmosphere_s();
    clouds_2D_settings pc = s_clouds2D();

    vec3 clouds_scattering;
    float transmittance = 1.0;

    vec3 up_position = vec3(0.0, 1.0, 0.0);
    vec3 view_position = vec3(0.0, ac.planet_radius + cameraPosition.y, 0.0);
    
    float VdotL = dot(position, direction);
    float VdotU = dot(position, up_position);

    vec2 sphere = ray_sphere_intersection(view_position, position, ac.planet_radius * 0.99995);
    vec2 sphere_min = ray_sphere_intersection(view_position, position, ac.planet_radius + pc.altitude_min);
    vec2 sphere_max = ray_sphere_intersection(view_position, position, ac.planet_radius + pc.altitude_max);

    if(transmittance <= pc.treshold) return vec4(0.0, 0.0, 0.0, 1.0);
    if((eyeAltitude < pc.altitude_min && sphere.y > 0.0) || (eyeAltitude > pc.altitude_max && position.y > 0)) return vec4(0.0, 0.0, 0.0, 1.0);

    vec3 clouds_position = position * sphere_min.y + cameraPosition;

    float density = planar_clouds_density(pc, clouds_position) * pc.thickness;
    if(density <= 0.0) return vec4(0.0, 0.0, 0.0, 1.0);

    float step_transmittance = exp2(-density);
    float step_transmittance_fraction = 1.0 - step_transmittance;

    float direct_transmittance = planar_clouds_transmittance(pc, clouds_position, direction, 32);
    float indirect_transmittance = planar_clouds_transmittance(pc, clouds_position, up_position, 8) * 0.1; 

    float powder = 1.0 - exp(-4.0 * density);

    float direct_powder = 1.0 - exp(-2.0 * direct_transmittance);
          direct_powder = mix(direct_powder, 4.0, phase_mie(VdotL, ac.mie_g));
          direct_powder = mix(direct_powder, 1.0, VdotL * 0.5 + 0.5) * powder;

    float indirect_powder = 1.0 - exp(-2.0 * indirect_transmittance);
          indirect_powder = mix(indirect_powder, 1.0, VdotU * 0.5 + 0.5) * powder;

    for(int i = 0; i < 8; ++i){
        float a = pow(0.6, i);
        float b = pow(0.4, i);
        float c = pow(0.8, i);

        float clouds_transmittance = transmittance * (step_transmittance_fraction * a);

        float step_direct_transmittance = clouds_transmittance * exp(-direct_transmittance * b);
        float step_indirect_transmittance = clouds_transmittance * exp(-indirect_transmittance * b);

        float phase0 = volumetric_clouds_phase(VdotL * c, direct_transmittance);
        float phase1 = 2.0 * volumetric_clouds_phase(VdotU * c, indirect_transmittance);

        clouds_scattering.x += step_direct_transmittance * direct_powder * phase0;
        clouds_scattering.y += step_indirect_transmittance * indirect_powder * phase1;
    }

    transmittance *= step_transmittance;
    transmittance = saturate((transmittance - pc.treshold) / (1.0 - pc.treshold));

    vec3 sunlight = atmospheric_transmittance(sun_direction, moon_direction);
    vec3 skylight = texture2(colortex1, project_sphere(up_position)).xyz * 4.5;

    vec3 scattering = clouds_scattering.x * sunlight;
         scattering += clouds_scattering.y * skylight;

    float dist = exp2(-length(clouds_position) * max3(ac.lambda_rayleigh));

    #if ATMOSPHERE_TYPE == 0
        return mix(vec4(scattering * 0.8, transmittance), vec4(atmosphere, 0.0), 1.0 - dist);
    #elif ATMOSPHERE_TYPE == 1
        return vec4(0.0, 0.0, 0.0, 1.0);
    #endif
}