/* RENDERTARGETS: 4 */
layout(location = 0) out vec3 skyCapture;

#include "/lib/head.glsl"

const vec2 viewSize     = vec2(512, 512);
const vec2 pixelSize    = 1.0 / viewSize;

in vec2 uv;

flat in mat2x3 skyColors;
flat in mat4x3 lightColor;

flat in vec3 sunDir;
flat in vec3 moonDir;

//uniform vec3 sunDir, moonDir;

uniform vec3 cloudLightDir;

uniform vec4 daytime;

uniform mat4 gbufferProjectionInverse, gbufferModelViewInverse;

#include "/lib/atmos/project.glsl"

float mieHG(float cosTheta, float g) {
    float mie   = 1.0 + sqr(g) - 2.0*g*cosTheta;
        mie     = (1.0 - sqr(g)) / ((4.0*pi) * mie*(mie*0.5+0.5));
    return mie;
}

vec3 skyGradient(vec3 direction) {
    float vDotS     = dot(direction, sunDir);
    float vDotM     = dot(direction, moonDir);

    float horizon   = exp(-max0(direction.y) * sqrPi);
        horizon    *= exp(-max0(-direction.y) * pi);

    float sunAmb    = saturate(1.35 * mieHG(vDotS, 0.2) / 0.2);

        horizon     = mix(horizon, horizon * sunAmb, sqr(daytime.x + daytime.z) * 0.8);

    float zenith    = exp(-max0(direction.y)) * 0.8 + 0.2;

    vec3 sunColor   = lightColor[0] * 0.2;

    float sunScatter = mieHG(vDotS, 0.78) * rpi * (1 - daytime.w);

    float glowAmount = cube(1 - linStep(direction.y, 0.0, 0.31)) * 0.96 * sunAmb;

    vec3 glowColor  = mix(skyColors[1], sunColor * skyColors[1] * 0.9 + sunColor * sunScatter, sstep(sunDir.y, -0.1, 0.01));
        glowColor   = mix(glowColor * 0.5, skyColors[0] * 0.5, 1.0 - exp(-max0(-direction.y - 0.06) * 0.71) * 0.8);

    vec3 sky        = skyColors[0] * zenith;
        sky         = mix(sky, glowColor, glowAmount);
        sky         = mix(sky, skyColors[1] * 1.2, horizon);
        sky        += sunColor * sunScatter * 1.35 * sqrt3;
        sky        += lightColor[1] * mieHG(vDotM, 0.74) * rpi * sqrt(daytime.w);

    return sky;
}

uniform sampler2D colortex7;
uniform sampler2D noisetex;

uniform int frameCounter;
uniform int worldTime;
uniform float worldAnimTime, cloudLightFlip, frameTimeCounter;

uniform vec3 cameraPosition;

flat in vec3 cloudSunlight;
flat in vec3 cloudSkylight;

#include "/lib/frag/bluenoise.glsl"
#include "/lib/atmos/clouds.glsl"

vec4 volumetricClouds(vec3 worldDir, float vDotL, float dither, vec3 skyColor) {
    vec3 totalScattering    = vec3(0.0);
    float totalTransmittance = 1.0;

    float eyeAltitude   = 64.0;
    vec3 cameraPos      = vec3(cameraPosition.x, 64.0, cameraPosition.z);

    vec3 sunlight       = (worldTime>23000 || worldTime<12900) ? cloudSunlight : lightColor[1];
        sunlight       *= cloudLightFlip / sqrt3;
    vec3 skylight       = mix(cloudSkylight, lightColor[1] * (1.0 - sqrt(daytime.w)*0.96), 0.1) / pi;

    float pFade         = saturate(mieHG(vDotL, 0.65));

    bool isBelowVol = eyeAltitude < cloudVol0MidY;
    bool visibleVol = worldDir.y > 0.0 && isBelowVol || worldDir.y < 0.0 && !isBelowVol;

    #ifdef cloudVolumeStoryMode
        const float sigmaA  = 0.1;
            sunlight       *= pi * sqrt2;
            skylight       /= sqrt2;
        const float sigmaT  = 1.25;
    #else
        const float sigmaA  = 0.2;
        const float sigmaT  = 1.125;
    #endif


    if (visibleVol) {
        vec3 bottom     = worldDir * ((cloudVolume0Alt - eyeAltitude) * rcp(worldDir.y));
        vec3 top        = worldDir * ((cloudVol0MaxY - eyeAltitude) * rcp(worldDir.y));

        if (worldDir.y < 0.0 && isBelowVol || worldDir.y > 0.0 && !isBelowVol) {
            bottom      = vec3(0.0);
            top         = vec3(0.0);
        }

        vec3 start      = isBelowVol ? bottom : top;
        vec3 end        = isBelowVol ? top : bottom;

        uint steps          = uint(cloudVolume0Samples);

        vec3 rStep          = (end - start) * rcp(float(steps));
        vec3 rPos           = rStep * dither + start + cameraPos;
        float rLength       = length(rStep);

        vec3 scattering     = vec3(0.0);
        float transmittance = 1.0;

        for (uint i = 0; i < steps; ++i, rPos += rStep) {
            if (transmittance < 0.01) break;
            if (rPos.y < cloudVolume0Alt || rPos.y > cloudVol0MaxY) continue;

            float dist  = distance(rPos, cameraPos);
            if (dist > cloudVolume0Clip) continue;

            float density = cloudVolume0Shape(rPos);
            if (density <= 0.0) continue;

            float extinction    = density * sigmaT;
            float stepT         = exp(-extinction * rLength);
            float integral      = (1.0 - stepT) * rcp(sigmaT);

            vec3 stepScatter    = vec3(0.0);

            float lightOD       = cloudVolume0LightOD(rPos, 5, cloudLightDir) * sigmaA;
            float skyOD         = cloudVolume0LightOD(rPos, 4, vec3(0.0, 1.0, 0.0)) * sigmaA;

            float powder        = 8.0 * (1.0 - 0.97 * exp(-extinction * 18.0));

            #ifdef cloudVolumeStoryMode
                float anisoPowder   = 1.0;
            #else
                float anisoPowder   = mix(powder, 1.0, pFade);
            #endif

            vec3 phaseG         = pow(vec3(0.45, 0.25, 0.95), vec3(1.0 + lightOD));

            float phase = cloudPhase(vDotL, 1.0, phaseG);

            stepScatter.x  += max(expf(-lightOD * sigmaT), expf(-lightOD * sigmaT * 0.2) * 0.75) * phase * anisoPowder * sigmaT;
            stepScatter.y  += max(expf(-skyOD * sigmaT), expf(-skyOD * sigmaT * 0.2) * 0.75) * powder * sigmaT;

            stepScatter     = (sunlight * stepScatter.x) + (skylight * stepScatter.y);

            float atmosFade = expf(-dist * 2.8e-3);

            stepScatter     = mix(skyColor * sigmaT, stepScatter, atmosFade);

            scattering     += stepScatter * (integral * transmittance);

            transmittance  *= stepT;
        }

        transmittance       = linStep(transmittance, 0.01, 1.0);

        totalScattering    += scattering;
        totalTransmittance *= transmittance;
    }


    #ifdef cloudVolume1Enabled
    isBelowVol = eyeAltitude < cloudVol1MidY;
    visibleVol = worldDir.y > 0.0 && isBelowVol || worldDir.y < 0.0 && !isBelowVol;

    if (visibleVol) {
        vec3 bottom     = worldDir * ((cloudVolume1Alt - eyeAltitude) * rcp(worldDir.y));
        vec3 top        = worldDir * ((cloudVol1MaxY - eyeAltitude) * rcp(worldDir.y));

        if (worldDir.y < 0.0 && isBelowVol || worldDir.y > 0.0 && !isBelowVol) {
            bottom      = vec3(0.0);
            top         = vec3(0.0);
        }

        vec3 start      = isBelowVol ? bottom : top;
        vec3 end        = isBelowVol ? top : bottom;

        uint steps          = uint(cloudVolume1Samples);

        vec3 rStep          = (end - start) * rcp(float(steps));
        vec3 rPos           = rStep * dither + start + cameraPos;
        float rLength       = length(rStep);

        vec3 scattering     = vec3(0.0);
        float transmittance = 1.0;

        for (uint i = 0; i < steps; ++i, rPos += rStep) {
            if (transmittance < 0.01) break;
            if (rPos.y < cloudVolume1Alt || rPos.y > cloudVol1MaxY) continue;

            float dist  = distance(rPos, cameraPos);
            if (dist > cloudVolume1Clip) continue;

            float density = cloudVolume1Shape(rPos);
            if (density <= 0.0) continue;

            float extinction    = density * sigmaT;
            float stepT         = exp(-extinction * rLength);
            float integral      = (1.0 - stepT) * rcp(sigmaT);

            vec3 stepScatter    = vec3(0.0);

            float lightOD       = cloudVolume1LightOD(rPos, 4, cloudLightDir) * sigmaA;
            float skyOD         = cloudVolume1LightOD(rPos, 4, vec3(0.0, 1.0, 0.0)) * sigmaA;

            float powder        = 8.0 * (1.0 - 0.97 * exp(-extinction * 18.0)); 

            #ifdef cloudVolumeStoryMode
                float anisoPowder   = 1.0;
            #else
                float anisoPowder   = mix(powder, 1.0, pFade);
            #endif

            vec3 phaseG         = pow(vec3(0.45, 0.25, 0.95), vec3(1.0 + lightOD));

            float phase = cloudPhase(vDotL, 1.0, phaseG);

            stepScatter.x  += max(expf(-lightOD * sigmaT), expf(-lightOD * sigmaT * 0.2) * 0.75) * phase * anisoPowder * sigmaT;
            stepScatter.y  += max(expf(-skyOD * sigmaT), expf(-skyOD * sigmaT * 0.2) * 0.75) * powder * sigmaT;

            stepScatter     = (sunlight * stepScatter.x) + (skylight * stepScatter.y);

            float atmosFade = expf(-dist * 2.8e-3 * 0.9);

            stepScatter     = mix(skyColor * sigmaT, stepScatter, atmosFade);

            scattering     += stepScatter * (integral * transmittance);

            transmittance  *= stepT;
        }

        transmittance       = linStep(transmittance, 0.01, 1.0);

        if (eyeAltitude < cloudVolume1Alt) {
            totalScattering    += scattering * totalTransmittance;
        } else {
            totalScattering     = totalScattering * transmittance + scattering;
        }

        totalTransmittance *= transmittance;
    }
    #endif

    vec4 result     = vec4(totalScattering, totalTransmittance);
        result      = mix(result, vec4(0,0,0,1), exp(-max0(worldDir.y * pi4)));

    return result;
}


void main() {
    vec2 projectionUV   = fract(uv * vec2(1.0, 2.0));

    if (uv.y < 0.5) {
        // Clear Sky Capture
        vec3 direction  = unprojectSky(projectionUV);

        skyCapture      = skyGradient(direction);
    } else {
        // Sky Capture with Clouds (for Reflections)
        vec3 direction  = unprojectSky(projectionUV);

        skyCapture      = skyGradient(direction);
        skyCapture     *= mix(exp(-max0(-direction.y) * cube(euler)), 1.0, 0.2);

        #ifdef cloudVolumeEnabled
        vec4 clouds     = volumetricClouds(direction, dot(direction, cloudLightDir), ditherBluenoiseStatic(),skyCapture);

        skyCapture      = skyCapture * clouds.a + clouds.rgb;
        #endif
    }
}