

struct GBufferData
{
	vec4 albedo;
	float depth;
	vec3 normal;
	vec2 mcLightmap;
	float smoothness;
	float metalness;
	float materialID;
	float emissive;
	float parallaxShadow;
	float rainMask;
};

struct GBufferDataTransparent
{
	vec4 albedo;
	float depth;
	vec3 normal;
	vec2 mcLightmap;
	float materialID;
};


#include "DataPacking.inc"



/*
[pack(albedo.xy), pack(albedo.z, emissive), pack(mclightmap.xy), albedo.a]
[pack(metal, smooth), pack(matID, parallaxShadow), encNormal.x, encNormal.y]
*/

// [pack(pack(mcLightmap.xy), matID), pack(albedo.r, albedo.g), pack(albedo.b, albedo.a), 0]
// [encNormal.xy, 0, 0]

//Output helpers
void OutputGBufferDataSolid(in GBufferData data, out vec4 target0, out vec4 target1, out vec4 target2)
{
	data.materialID = (data.materialID + 0.1) / 255.0;



	//gcolor RGBA16
	target0 = vec4
	(
		data.albedo.rgba
	);

	//gdepth RGBA16
	target1 = vec4
	(
		PackTwo8BitTo16Bit(data.metalness, data.smoothness),
		PackTwo8BitTo16Bit(data.materialID, data.parallaxShadow),
		EncodeNormal(data.normal)
	);

	//gnormal RGBA16
	target2 = vec4
	(
		data.mcLightmap.xy,
		data.emissive,
		0.0 	//rain mask
	);
}

void OutputGBufferDataTransparent(in GBufferDataTransparent data, out vec4 target0, out vec4 target1)
{
	data.materialID = (data.materialID + 0.1) / 255.0;

	//composite RGBA16
	target0 = vec4
	(
		PackTwo8BitTo16Bit(PackTwo4BitTo8Bit(data.mcLightmap.x, data.mcLightmap.y), data.materialID),
		PackTwo8BitTo16Bit(data.albedo.r, data.albedo.g),
		PackTwo8BitTo16Bit(data.albedo.b, data.albedo.a),
		1.0
	);

	//gaux1 RGBA16
	target1 = vec4
	(
		EncodeNormal(data.normal),
		1.0,
		1.0
	);
}


GBufferData GetGBufferData()
{
	GBufferData data;

	vec4 tex0 = texture2DLod(gcolor, texcoord.st, 0);
	vec4 tex1 = texture2DLod(gdepth, texcoord.st, 0);
	vec4 tex2 = texture2DLod(gnormal, texcoord.st, 0);

	float depthTex = texture2D(depthtex1, texcoord.st).x;

	vec2 unpacked1x = UnpackTwo8BitFrom16Bit(tex1.x);
	vec2 unpacked1y = UnpackTwo8BitFrom16Bit(tex1.y);

	data.albedo.rgb = vec3(tex0.xyz);
	data.albedo.rgb = GammaToLinear(data.albedo.rgb);
	data.albedo.a = 1.0;

	data.mcLightmap = tex2.xy;
	data.mcLightmap.g = CurveBlockLightSky(data.mcLightmap.g);
	data.mcLightmap.r = CurveBlockLightTorch(data.mcLightmap.r);

	data.normal = DecodeNormal(tex1.zw);

	data.smoothness = unpacked1x.y;
	data.metalness = unpacked1x.x;
	data.emissive = tex2.y;

	data.materialID = unpacked1y.x * 255.0;

	data.depth = depthTex;

	data.parallaxShadow = unpacked1y.y;

	data.rainMask = tex2.a;


	return data;
}

GBufferDataTransparent GetGBufferDataTransparent()
{
	GBufferDataTransparent data;

	vec4 tex2 = texture2DLod(composite, texcoord.st, 0);
	vec4 tex3 = texture2DLod(gaux1, texcoord.st, 0);

	vec2 unpacked2x = UnpackTwo8BitFrom16Bit(tex2.x);
	vec2 unpacked2y = UnpackTwo8BitFrom16Bit(tex2.y);
	vec2 unpacked2z = UnpackTwo8BitFrom16Bit(tex2.z);
	vec2 unpacked2xx = UnpackTwo4BitFrom8Bit(unpacked2x.x);



	data.albedo = vec4(unpacked2y.xy, unpacked2z.xy);
	data.albedo.rgb = GammaToLinear(data.albedo.rgb);

	data.mcLightmap = vec2(unpacked2xx.xy);
	data.mcLightmap.g = CurveBlockLightSky(saturate(data.mcLightmap.g * 1.1));
	data.mcLightmap.r = CurveBlockLightTorch(data.mcLightmap.r);

	data.materialID = unpacked2x.y * 255.0;

	data.normal = DecodeNormal(tex3.xy);

	data.depth = texture2D(gdepthtex, texcoord.st).x;

	return data;
}



//Individual getters


vec3 GetNormals(vec2 coord)
{
	return DecodeNormal(texture2DLod(gdepth, coord, 0).zw);
}

float GetDepthLinear(in vec2 coord) 
{					
	return (near * far) / (texture2DLod(depthtex1, coord, 0).x * (near - far) + far);
}

float GetDepth(vec2 coord)
{
	return texture2DLod(depthtex1, coord, 0).x;
}

float GetMaterialIDs(vec2 coord)
{
	return  UnpackTwo8BitFrom16Bit(ReadBufferRaw(gdepth, coord).y).x * 255.0;
}


vec2 GetMCLightmapRaw(vec2 coord)
{
	vec2 lightmap = UnpackTwo8BitFrom16Bit(ReadBufferRaw(gcolor, coord).z);

	return lightmap;
}

vec2 GetMCLightmap(vec2 coord)
{
	vec2 lightmap = GetMCLightmapRaw(coord);
	lightmap.g = CurveBlockLightSky(lightmap.g);
	lightmap.r = CurveBlockLightTorch(lightmap.r);

	return lightmap;
}






vec3  	GetWaterNormals(in vec2 coord) {				//Function that retrieves the screen space surface normals. Used for lighting calculations
	return DecodeNormal(texture2DLod(gaux1, coord, 0).xy);
}

float GetTransparentMaterialIDs(vec2 coord)
{
	return UnpackTwo8BitFrom16Bit(ReadBufferRaw(composite, coord).x).y * 255.0;
}

vec2 GetTransparentMCLightmapRaw(vec2 coord)
{
	return UnpackTwo4BitFrom8Bit(UnpackTwo8BitFrom16Bit(ReadBufferRaw(composite, coord).x).x).xy;
}

vec2 GetTransparentMCLightmap(vec2 coord)
{
	vec2 lightmap = GetTransparentMCLightmapRaw(coord);
	lightmap.g = CurveBlockLightSky(lightmap.g);
	lightmap.r = CurveBlockLightTorch(lightmap.r);

	return lightmap;
}

vec4 GetTransparentAlbedo(vec2 coord)
{
	vec4 data = ReadBufferRaw(gnormal, coord);

	vec4 albedo;
	albedo.xy = UnpackTwo8BitFrom16Bit(data.y).xy;
	albedo.zw = UnpackTwo8BitFrom16Bit(data.z).xy;

	return albedo;
}






























struct MaterialMask
{
	float sky;
	float land;
	float grass;
	float leaves;
	float hand;
	float entityPlayer;
	float water;
	float stainedGlass;
	float ice;
	float torch;
	float lava;
	float glowstone;
	float fire;
};





//Material mask stuff
float GetMaterialMask(const in int ID, in float matID) 
{
	//Catch last part of sky
	if (matID > 254.0f) 
	{
		matID = 0.0f;
	}

	// if (matID == ID) 
	if (abs(matID - ID) < 0.5) 
	{
		return 1.0f;
	} 
	else 
	{
		return 0.0f;
	}
}

float GetMaterialMask(vec2 coord, const in int ID)
{
	float matID = GetMaterialIDs(coord);
	return GetMaterialMask(ID, matID);
}

bool 	GetSkyMask(in vec2 coord)
{
	float matID = GetMaterialIDs(coord);
	matID = floor(matID * 255.0f);

	if (matID < 1.0f || matID > 254.0f)
	{
		return true;
	} else {
		return false;
	}
}

MaterialMask CalculateMasks(float materialID)
{
	MaterialMask mask;


	// if (isEyeInWater > 0)
	// 	mask.sky = 0.0f;
	// else
	// {
	// 	mask.sky = 0.0;
	// 	if (texture2D(depthtex1, texcoord.st).x > 0.999999)
	// 	{
	// 		mask.sky = 1.0;
	// 	}
	// }
		mask.sky = GetMaterialMask(0, materialID);
		//mask.sky = texture2D(depthtex1, texcoord).x > 0.999999 ? 1.0 : 0.0;



	mask.land 			= GetMaterialMask(1, materialID);
	mask.grass 			= GetMaterialMask(2, materialID);
	mask.leaves 		= GetMaterialMask(3, materialID);
	mask.hand 			= GetMaterialMask(4, materialID);
	mask.entityPlayer 	= GetMaterialMask(5, materialID);
	mask.water 			= GetMaterialMask(6, materialID);
	mask.stainedGlass	= GetMaterialMask(7, materialID);
	mask.ice 			= GetMaterialMask(8, materialID);

	mask.torch 			= GetMaterialMask(30, materialID);
	mask.lava 			= GetMaterialMask(31, materialID);
	mask.glowstone 		= GetMaterialMask(32, materialID);
	mask.fire 			= GetMaterialMask(33, materialID);

	return mask;
}
