
layout(location = 0) out vec4 skyboxData;
layout(location = 1) out vec4 colortex5Out;

/* DRAWBUFFERS:25 */

//in vec2 texcoord;

flat in vec3 colorSunlight;
flat in vec3 colorSkylight;

#include "/lib/Head/Common.inc"

uniform sampler2D noisetex;
uniform sampler2D depthtex1;

uniform float nightVision;
uniform float viewWidth;
uniform float viewHeight;
uniform float wetness;
uniform float eyeAltitude;
uniform float isLightningFlashing;
uniform float worldTimeCounter;

uniform vec3 cameraPosition;
uniform vec3 worldSunVector;
uniform vec3 worldLightVector;

uniform int frameCounter;
uniform int moonPhase;

uniform bool cloudMoonlit;

uniform mat4 gbufferProjectionInverse;
uniform mat4 gbufferModelViewInverse;

uniform vec2 screenPixelSize;
uniform vec2 screenSize;

vec3 ScreenToViewSpaceRaw(in vec3 screenPos) {	
	vec3 NDCPos = screenPos * 2.0 - 1.0;
	vec4 viewPos = gbufferProjectionInverse * vec4(NDCPos, 1.0);
		 viewPos /= viewPos.w;

	return viewPos.xyz;
}

#include "/lib/Head/Noise.inc"

#include "/lib/Atmosphere/Atmosphere.glsl"

#include "/lib/Atmosphere/VolumetricClouds.glsl"

#include "/lib/Atmosphere/PlanarClouds.glsl"

#ifdef AURORA
	#include "/lib/Atmosphere/Aurora.glsl"
#endif
/*
float depthMax2x2(in vec2 coord) {
	vec4 depthSamples0 = textureGather(depthtex0, coord);
	return maxOf(depthSamples0);
}
*/
float depthMax4x4(in vec2 coord) {
	vec4 depthSamples0 = textureGather(depthtex1, coord + vec2( 2.0,  2.0) * screenPixelSize);
	vec4 depthSamples1 = textureGather(depthtex1, coord + vec2(-2.0,  2.0) * screenPixelSize);
	vec4 depthSamples2 = textureGather(depthtex1, coord + vec2( 2.0, -2.0) * screenPixelSize);
	vec4 depthSamples3 = textureGather(depthtex1, coord + vec2(-2.0, -2.0) * screenPixelSize);

	return max(
		max(maxOf(depthSamples0), maxOf(depthSamples1)),
		max(maxOf(depthSamples2), maxOf(depthSamples3))
	);
}

#if TEMPORAL_UPSCALING == 2
	const ivec2[4] checkerboardOffset = ivec2[4](
		ivec2(0, 0), ivec2(1, 1),
		ivec2(1, 0), ivec2(0, 1)
	);
#elif TEMPORAL_UPSCALING == 3
	const ivec2[9] checkerboardOffset = ivec2[9](
		ivec2(0, 0), ivec2(2, 0), ivec2(0, 2),
		ivec2(2, 2), ivec2(1, 1), ivec2(1, 0),
		ivec2(1, 2), ivec2(0, 1), ivec2(2, 1)
	);
#elif TEMPORAL_UPSCALING == 4
	const ivec2[16] checkerboardOffset = ivec2[16](
		ivec2(0, 0), ivec2(2, 0), ivec2(0, 2), ivec2(2, 2),
		ivec2(1, 1), ivec2(3, 1), ivec2(1, 3), ivec2(3, 3),
		ivec2(1, 0), ivec2(3, 0), ivec2(1, 2), ivec2(3, 2),
		ivec2(0, 1), ivec2(2, 1), ivec2(0, 3), ivec2(2, 3)
	);
#endif

const int cloudsRenderFactor = TEMPORAL_UPSCALING * TEMPORAL_UPSCALING;

// float InterleavedGradientNoiseCheckerboard(in vec2 coord) {
// 	return fract(52.9829189 * fract(0.06711056 * coord.x + 0.00583715 * coord.y + 0.00623715 * ((frameCounter / cloudsRenderFactor) & 63)));
// }

vec4 RenderSkybox(in vec3 worldDir, in float dither/* , in float lightNoise */, in CloudProperties cloudProperties) {
	vec4 cloudsData = vec4(0.0, 0.0, 0.0, 1.0);
	vec3 skyColor = SkyShading(worldDir);

	float LdotV = dot(worldDir, worldLightVector);

	vec4 phases;	/* forwardsLobe */										/* backwardsLobe */																	/* forwardsPeak */
	phases.x = 	HenyeyGreensteinPhase(LdotV, cloudForwardG) 	  * 0.7  + HenyeyGreensteinPhase(LdotV, cloudBackwardG)		  * cloudBackwardWeight  	  + CornetteShanksPhase(LdotV, 0.9) * cloudProperties.cloudPeakWeight;
	phases.y = 	HenyeyGreensteinPhase(LdotV, cloudForwardG * 0.7) * 0.35 + HenyeyGreensteinPhase(LdotV, cloudBackwardG * 0.7) * cloudBackwardWeight * 0.6 + CornetteShanksPhase(LdotV, 0.6) * cloudProperties.cloudPeakWeight * 0.5;
	phases.z = 	HenyeyGreensteinPhase(LdotV, cloudForwardG * 0.5) * 0.17 + HenyeyGreensteinPhase(LdotV, cloudBackwardG * 0.5) * cloudBackwardWeight * 0.3 + CornetteShanksPhase(LdotV, 0.4) * cloudProperties.cloudPeakWeight * 0.2;
	phases.w = 	HenyeyGreensteinPhase(LdotV, cloudForwardG * 0.3) * 0.08 + HenyeyGreensteinPhase(LdotV, cloudBackwardG * 0.3) * cloudBackwardWeight * 0.2 + CornetteShanksPhase(LdotV, 0.2) * cloudProperties.cloudPeakWeight * 0.1;
	//phases.x = MiePhaseClouds(LdotV, vec3(cloudForwardG, cloudBackwardG, 0.9), vec3(0.7, cloudBackwardWeight, cloudPeakWeight));
	//phases.y = MiePhaseClouds(LdotV, vec3(cloudForwardG, cloudBackwardG, 0.9) * 0.7, vec3(0.35, cloudBackwardWeight * 0.6, cloudPeakWeight * 0.5));
	//phases.z = MiePhaseClouds(LdotV, vec3(cloudForwardG, cloudBackwardG, 0.9) * 0.5, vec3(0.17, cloudBackwardWeight * 0.3, cloudPeakWeight * 0.2));
	//phases.w = MiePhaseClouds(LdotV, vec3(cloudForwardG, cloudBackwardG, 0.9) * 0.3, vec3(0.08, cloudBackwardWeight * 0.15, cloudPeakWeight * 0.1));

	#ifdef VOLUMETRIC_CLOUDS
		if ((worldDir.y > 0.0 && eyeAltitude < cloudProperties.altitude)
		|| (worldDir.y < 0.0 && eyeAltitude > cloudProperties.maxAltitude)) {

			vec3 planeOrigin = vec3(0.0, planetRadius + eyeAltitude, 0.0);
			vec2 bottomIntersection = RaySphereIntersection(planeOrigin, worldDir, planetRadius + cloudProperties.altitude);
			vec2 topIntersection = RaySphereIntersection(planeOrigin, worldDir, planetRadius + cloudProperties.maxAltitude);

			float startLength, endLength;
			if (eyeAltitude > cloudProperties.maxAltitude) {
				startLength = topIntersection.x;
				endLength = bottomIntersection.x;
			} else {
				startLength = bottomIntersection.y;
				endLength = topIntersection.y;
			}

			// The range of eye in cloudsData
			float rayRange = oneMinus(saturate((eyeAltitude - cloudProperties.maxAltitude) * 0.1)) *
							oneMinus(saturate((cloudProperties.altitude - eyeAltitude) * 0.1));

			// The ray distance in range
			float rayDist = bottomIntersection.y >= 0.0 && eyeAltitude > cloudProperties.altitude ? bottomIntersection.x : topIntersection.y;
			//rayDist = min(rayDist, cloudProperties.altitude * 10.0);

			startLength *= oneMinus(rayRange);
			endLength = mix(endLength, rayDist, rayRange);

			uint raySteps = CLOUD_CUMULUS_SAMPLES;
			raySteps = uint(mix(raySteps, uint(raySteps / 1.6), abs(worldDir.y))); // Steps Fade

			float rayLength = clamp(endLength - startLength, 0.0, 2e4) * rcp(raySteps);
			vec3 rayStep = rayLength * worldDir;
			vec3 rayPos = (startLength + rayLength * dither) * worldDir + cameraPosition;

			//float dist = distance(rayPos, cameraPosition);
			//float noiseDetail = mix(GetNoiseDetail(worldDir), 1.0, exp2(-dist * 0.0007f) * 0.8) * 0.03;
			float noiseDetail = GetNoiseDetail(worldDir);
			//float lightNoise = InterleavedGradientNoise();

			//uint raySteps = max(uint(CLOUD_CUMULUS_SAMPLES - sqrt(dist) * 0.06), CLOUD_CUMULUS_SAMPLES / 2);
			//uint raySteps = uint(CLOUD_CUMULUS_SAMPLES);

			//float LdotV01 = dot(worldDir, worldLightVector) * 0.5 + 0.5;
			//float LdotV = dot(worldDir, worldLightVector);

			float scatteringSun = 0.0;
			float scatteringSky = 0.0;

			//float powderIntensity = saturate(CornetteShanksPhase(LdotV, 0.5));
			//float powderIntensity = 0.8 * sqr(dot(worldDir, worldLightVector) * 0.5 + 0.5);
			float transmittance = 1.0;

			for (uint i = 0u; i < raySteps; ++i, rayPos += rayStep) {
				if (transmittance < minTransmittance) break;
				if (rayPos.y < cloudProperties.altitude || rayPos.y > cloudProperties.maxAltitude) continue;

				float dist = distance(rayPos, cameraPosition);
				if (dist > 1e5 - 3e4 * wetness) continue;

				float density = CloudVolumeDensity(cloudProperties, rayPos, 5u, mix(noiseDetail, 1.0, exp2(-dist * 0.001f)));
				if (density < 1e-4) continue;

				vec2 lightNoise = hash2(fract(rayPos)) * 0.4 + 0.3;

				float sunlightOD = CloudVolumeSunLightOD(cloudProperties, rayPos, lightNoise.x);

				//float powder = 1.0 - expf(-density * 32.0);
				//powder = powder * oneMinus(powderIntensity) + powderIntensity;
				float bounceEstimate = oneMinus(expf(-density * 32.0)) * 0.82;
				bounceEstimate /= 1.0 - bounceEstimate;

				//float sunlightEnergy = 	expf(-sunlightOD * 2.0) * phases.x;
				//sunlightEnergy += 		expf(-sunlightOD * 0.8) * phases.y;
				//sunlightEnergy += 		expf(-sunlightOD * 0.3) * phases.z;
				//sunlightEnergy += 		expf(-sunlightOD * 0.1) * phases.w;
				float sunlightEnergy = 	rcp(sunlightOD * 2.0 + 1.0) * phases.x;
				sunlightEnergy += 		rcp(sunlightOD * 0.9 + 1.0) * phases.y;
				sunlightEnergy += 		rcp(sunlightOD * 0.4 + 1.0) * phases.z;
				sunlightEnergy += 		rcp(sunlightOD * 0.2 + 1.0) * phases.w;

				float skylightEnergy = CloudVolumeSkyLightOD(cloudProperties, rayPos, lightNoise.y);
				skylightEnergy = expf(-skylightEnergy) + expf(-skylightEnergy * 0.1) * 0.1;

				float stepTransmittance = expf(-density * sqrt(density) * 0.4 * rayLength);
				float cloudsTemp = bounceEstimate * transmittance * oneMinus(stepTransmittance);
				scatteringSun += sunlightEnergy * cloudsTemp;
				scatteringSky += skylightEnergy * cloudsTemp;
				transmittance *= stepTransmittance;	
			}

			if (transmittance < 1.0 - minTransmittance) {
				vec3 scattering = scatteringSun * 48.0 * cloudProperties.sunlighting * SunAbsorptionAtAltitude(1.0);
				scattering *= saturate(worldLightVector.y * 40.0);
				if (cloudMoonlit) scattering = DoNightEye(scattering * MoonFlux);
				scattering += scatteringSky * 0.15 * cloudProperties.skylighting * colorSkylight;

				// lightning
				if (isLightningFlashing > 1e-2) scattering += sqr(scatteringSky) * 0.1 * lightningColor;

				#ifdef AURORA
					scattering += sqr(scatteringSky) * vec3(0.0, 0.002, 0.001) * auroraAmount;
				#endif

				float atmosFade = expf(-distance(rayPos, cameraPosition) * (0.03 + 0.02 * wetness) * rcp(cloudProperties.altitude));
				//color *= 1.0 - atmosFade + transmittance * atmosFade;

				scattering += Atmosphere(worldDir, worldSunVector, 1.0, endLength * 7e-4) * 0.4;
				//atmos += DoNightEye(Atmosphere(worldDir, -worldSunVector, 1.0, endLength * 1e-4) * MoonFlux);
				//color += (scattering + atmos) * atmosFade;
				//color += scattering * atmosFade;
				scattering *= atmosFade;
				scattering += skyColor * oneMinus(transmittance) * oneMinus(atmosFade);

				cloudsData = vec4(scattering, transmittance);
			}
		}
	#endif

	#ifdef PLANAR_CLOUDS
		//vec4 cloudsTemp = PlanarClouds(worldDir, dither, phases, transmittanceTemp);

		if ((worldDir.y > 0.0 && eyeAltitude < CLOUD_PLANE_ALTITUDE)
		|| (worldDir.y < 0.0 && eyeAltitude > CLOUD_PLANE_ALTITUDE)) {
			vec3 planeOrigin = vec3(0.0, planetRadius + eyeAltitude, 0.0);
			vec2 intersection = RaySphereIntersection(planeOrigin, worldDir, planetRadius + CLOUD_PLANE_ALTITUDE);
			float cloudDistance = eyeAltitude > CLOUD_PLANE_ALTITUDE ? intersection.x : intersection.y;

			if (cloudDistance > 0.0) {
				vec3 cloudPos = worldDir * cloudDistance;

				//float cloudDist = length(cloudPos);
				if (cloudDistance < 2e5 - 5e4 * wetness) {
					//float LdotV = dot(worldDir, worldLightVector);

					vec3 atmos = Atmosphere(worldDir, worldSunVector, 1.0, cloudDistance * 7e-4) * 0.4;
					//atmos += DoNightEye(Atmosphere(worldDir, -worldSunVector, 1.0, cloudDist * 2e-4) * MoonFlux);
					vec4 cloudsTemp = vec4(0.0, 0.0, 0.0, 1.0);

					#ifdef CIRROCUMULUS_CLOUDS
						#ifdef CLOUDS_WEATHER
							if (cloudDynamicCovA < 0.4)
						#endif
						{
							vec4 sampleTemp = PlanarSample1(cloudDistance, cameraPosition.xz + cloudPos.xz, LdotV, dither, phases, worldDir);

							if (sampleTemp.a > minTransmittance) {
								float atmosFade = expf(-cloudDistance * fma(0.02, wetness, 0.12) * rcp(float(CLOUD_PLANE_ALTITUDE)));
								sampleTemp.rgb += atmos * sampleTemp.a;
								sampleTemp.rgb *= atmosFade;
								sampleTemp.rgb += skyColor * sampleTemp.a * oneMinus(atmosFade);
							}

							cloudsTemp.rgb = sampleTemp.rgb;
							cloudsTemp.a -= sampleTemp.a;
						}
					#endif
					#if CIRRUS_CLOUDS > 0
						#ifdef CLOUDS_WEATHER
							if (cloudDynamicCovB < 0.5)
						#endif
						{
							vec4 sampleTemp = PlanarSample0(cloudDistance, cameraPosition.xz + cloudPos.xz, LdotV);

							if (sampleTemp.a > minTransmittance) {
								float atmosFade = expf(-cloudDistance * fma(0.02, wetness, 0.12) * rcp(float(CLOUD_PLANE_ALTITUDE)));
								sampleTemp.rgb += atmos * sampleTemp.a;
								sampleTemp.rgb *= atmosFade;
								sampleTemp.rgb += skyColor * sampleTemp.a * oneMinus(atmosFade);
							}

							cloudsTemp.rgb += sampleTemp.rgb * cloudsTemp.a;
							cloudsTemp.a *= 1.0 - sampleTemp.a;
						}
					#endif

					if (eyeAltitude < CLOUD_PLANE_ALTITUDE) {
						cloudsData.rgb += cloudsTemp.rgb * cloudsData.a;
					} else {
						cloudsData.rgb = cloudsData.rgb * cloudsTemp.a + cloudsTemp.rgb;
					}

					cloudsData.a *= cloudsTemp.a;
				}
			}
		}
	#endif

	vec4 skyboxData;
	skyboxData.rgb = skyColor * cloudsData.a + cloudsData.rgb;

	#ifdef AURORA
		if (auroraAmount > 1e-2) skyboxData.rgb += NightAurora(worldDir) * cloudsData.a;
	#endif

	skyboxData.a = remap(minTransmittance, 1.0, cloudsData.a);

	return clamp16F(skyboxData);
}

vec4 RenderSkybox(in vec3 worldDir, in CloudProperties cloudProperties) {
	vec4 cloudsData = vec4(0.0, 0.0, 0.0, 1.0);
	vec3 skyColor = SkyShading(worldDir);

	float LdotV = dot(worldDir, worldLightVector);

	vec4 phases;	/* forwardsLobe */										/* backwardsLobe */																	/* forwardsPeak */
	phases.x = 	HenyeyGreensteinPhase(LdotV, cloudForwardG) 	  * 0.7  + HenyeyGreensteinPhase(LdotV, cloudBackwardG)		  * cloudBackwardWeight  	  + CornetteShanksPhase(LdotV, 0.9) * cloudProperties.cloudPeakWeight;
	phases.y = 	HenyeyGreensteinPhase(LdotV, cloudForwardG * 0.7) * 0.35 + HenyeyGreensteinPhase(LdotV, cloudBackwardG * 0.7) * cloudBackwardWeight * 0.6 + CornetteShanksPhase(LdotV, 0.6) * cloudProperties.cloudPeakWeight * 0.5;
	phases.z = 	HenyeyGreensteinPhase(LdotV, cloudForwardG * 0.5) * 0.17 + HenyeyGreensteinPhase(LdotV, cloudBackwardG * 0.5) * cloudBackwardWeight * 0.3 + CornetteShanksPhase(LdotV, 0.4) * cloudProperties.cloudPeakWeight * 0.2;
	phases.w = 	HenyeyGreensteinPhase(LdotV, cloudForwardG * 0.3) * 0.08 + HenyeyGreensteinPhase(LdotV, cloudBackwardG * 0.3) * cloudBackwardWeight * 0.2 + CornetteShanksPhase(LdotV, 0.2) * cloudProperties.cloudPeakWeight * 0.1;
	//phases.x = MiePhaseClouds(LdotV, vec3(cloudForwardG, cloudBackwardG, 0.9), vec3(0.7, cloudBackwardWeight, cloudPeakWeight));
	//phases.y = MiePhaseClouds(LdotV, vec3(cloudForwardG, cloudBackwardG, 0.9) * 0.7, vec3(0.35, cloudBackwardWeight * 0.6, cloudPeakWeight * 0.5));
	//phases.z = MiePhaseClouds(LdotV, vec3(cloudForwardG, cloudBackwardG, 0.9) * 0.5, vec3(0.17, cloudBackwardWeight * 0.3, cloudPeakWeight * 0.2));
	//phases.w = MiePhaseClouds(LdotV, vec3(cloudForwardG, cloudBackwardG, 0.9) * 0.3, vec3(0.08, cloudBackwardWeight * 0.15, cloudPeakWeight * 0.1));

	#ifdef VOLUMETRIC_CLOUDS
		if ((worldDir.y > 0.0 && eyeAltitude < cloudProperties.altitude)
		|| (worldDir.y < 0.0 && eyeAltitude > cloudProperties.maxAltitude)) {

			vec3 planeOrigin = vec3(0.0, planetRadius + eyeAltitude, 0.0);
			vec2 bottomIntersection = RaySphereIntersection(planeOrigin, worldDir, planetRadius + cloudProperties.altitude);
			vec2 topIntersection = RaySphereIntersection(planeOrigin, worldDir, planetRadius + cloudProperties.maxAltitude);

			float startLength, endLength;
			if (eyeAltitude > cloudProperties.maxAltitude) {
				startLength = topIntersection.x;
				endLength = bottomIntersection.x;
			} else {
				startLength = bottomIntersection.y;
				endLength = topIntersection.y;
			}

			// The range of eye in cloudsData
			float rayRange = oneMinus(saturate((eyeAltitude - cloudProperties.maxAltitude) * 0.1)) *
							oneMinus(saturate((cloudProperties.altitude - eyeAltitude) * 0.1));

			// The ray distance in range
			float rayDist = bottomIntersection.y >= 0.0 && eyeAltitude > cloudProperties.altitude ? bottomIntersection.x : topIntersection.y;
			//rayDist = min(rayDist, cloudProperties.altitude * 10.0);

			startLength *= oneMinus(rayRange);
			endLength = mix(endLength, rayDist, rayRange);

			uint raySteps = CLOUD_CUMULUS_SAMPLES;
			raySteps = uint(mix(raySteps, uint(raySteps / 1.6), abs(worldDir.y))); // Steps Fade

			float rayLength = clamp(endLength - startLength, 0.0, 2e4) * rcp(raySteps);
			vec3 rayStep = rayLength * worldDir;
			vec3 rayPos = (startLength + rayLength * 0.5) * worldDir + cameraPosition;

			//float dist = distance(rayPos, cameraPosition);
			//float noiseDetail = mix(GetNoiseDetail(worldDir), 1.0, exp2(-dist * 0.0007f) * 0.8) * 0.03;
			float noiseDetail = GetNoiseDetail(worldDir);
			//float lightNoise = InterleavedGradientNoise();

			//uint raySteps = max(uint(CLOUD_CUMULUS_SAMPLES - sqrt(dist) * 0.06), CLOUD_CUMULUS_SAMPLES / 2);
			//uint raySteps = uint(CLOUD_CUMULUS_SAMPLES);

			//float LdotV01 = dot(worldDir, worldLightVector) * 0.5 + 0.5;
			//float LdotV = dot(worldDir, worldLightVector);

			float scatteringSun = 0.0;
			float scatteringSky = 0.0;

			//float powderIntensity = saturate(CornetteShanksPhase(LdotV, 0.5));
			//float powderIntensity = 0.8 * sqr(dot(worldDir, worldLightVector) * 0.5 + 0.5);
			float transmittance = 1.0;

			for (uint i = 0u; i < raySteps; ++i, rayPos += rayStep) {
				if (transmittance < minTransmittance) break;
				if (rayPos.y < cloudProperties.altitude || rayPos.y > cloudProperties.maxAltitude) continue;

				float dist = distance(rayPos, cameraPosition);
				if (dist > 1e5 - 3e4 * wetness) continue;

				float density = CloudVolumeDensity(cloudProperties, rayPos, 5u, mix(noiseDetail, 1.0, exp2(-dist * 0.001f)));
				if (density < 1e-4) continue;

				float sunlightOD = CloudVolumeSunLightOD(cloudProperties, rayPos, 0.5);

				//float powder = 1.0 - expf(-density * 32.0);
				//powder = powder * oneMinus(powderIntensity) + powderIntensity;
				float bounceEstimate = oneMinus(expf(-density * 32.0)) * 0.82;
				bounceEstimate /= 1.0 - bounceEstimate;

				//float sunlightEnergy = 	expf(-sunlightOD * 2.0) * phases.x;
				//sunlightEnergy += 		expf(-sunlightOD * 0.8) * phases.y;
				//sunlightEnergy += 		expf(-sunlightOD * 0.3) * phases.z;
				//sunlightEnergy += 		expf(-sunlightOD * 0.1) * phases.w;
				float sunlightEnergy = 	rcp(sunlightOD * 2.0 + 1.0) * phases.x;
				sunlightEnergy += 		rcp(sunlightOD * 0.9 + 1.0) * phases.y;
				sunlightEnergy += 		rcp(sunlightOD * 0.4 + 1.0) * phases.z;
				sunlightEnergy += 		rcp(sunlightOD * 0.2 + 1.0) * phases.w;

				float skylightEnergy = CloudVolumeSkyLightOD(cloudProperties, rayPos, 0.5);
				skylightEnergy = expf(-skylightEnergy) + expf(-skylightEnergy * 0.1) * 0.1;

				float stepTransmittance = expf(-density * 0.2 * rayLength);
				float cloudsTemp = bounceEstimate * transmittance * oneMinus(stepTransmittance);
				scatteringSun += sunlightEnergy * cloudsTemp;
				scatteringSky += skylightEnergy * cloudsTemp;
				transmittance *= stepTransmittance;	
			}

			if (transmittance < 1.0 - minTransmittance) {
				vec3 scattering = scatteringSun * 48.0 * cloudProperties.sunlighting * SunAbsorptionAtAltitude(1.0);
				scattering *= saturate(worldLightVector.y * 40.0);
				if (cloudMoonlit) scattering = DoNightEye(scattering * MoonFlux);
				scattering += scatteringSky * 0.1 * cloudProperties.skylighting * colorSkylight;

				// lightning
				if (isLightningFlashing > 1e-2) scattering += sqr(scatteringSky) * 0.1 * lightningColor;

				#ifdef AURORA
					scattering += sqr(scatteringSky) * vec3(0.0, 0.002, 0.001) * auroraAmount;
				#endif

				float atmosFade = expf(-distance(rayPos, cameraPosition) * (0.03 + 0.02 * wetness) * rcp(cloudProperties.altitude));
				//color *= 1.0 - atmosFade + transmittance * atmosFade;

				scattering += Atmosphere(worldDir, worldSunVector, 1.0, endLength * 7e-4) * 0.4;
				//atmos += DoNightEye(Atmosphere(worldDir, -worldSunVector, 1.0, endLength * 1e-4) * MoonFlux);
				//color += (scattering + atmos) * atmosFade;
				//color += scattering * atmosFade;
				scattering *= atmosFade;
				scattering += skyColor * oneMinus(transmittance) * oneMinus(atmosFade);

				cloudsData = vec4(scattering, transmittance);
			}
		}
	#endif

	#ifdef PLANAR_CLOUDS
		//vec4 cloudsTemp = PlanarClouds(worldDir, dither, phases, transmittanceTemp);

		if ((worldDir.y > 0.0 && eyeAltitude < CLOUD_PLANE_ALTITUDE)
		|| (worldDir.y < 0.0 && eyeAltitude > CLOUD_PLANE_ALTITUDE)) {
			float transmittanceTemp = 1.0;

			vec3 planeOrigin = vec3(0.0, planetRadius + eyeAltitude, 0.0);
			vec2 intersection = RaySphereIntersection(planeOrigin, worldDir, planetRadius + CLOUD_PLANE_ALTITUDE);
			float cloudDistance = eyeAltitude > CLOUD_PLANE_ALTITUDE ? intersection.x : intersection.y;

			if (cloudDistance > 0.0) {
				vec3 cloudPos = worldDir * cloudDistance;

				//float cloudDist = length(cloudPos);
				if (cloudDistance < 2e5 - 5e4 * wetness) {
					//float LdotV = dot(worldDir, worldLightVector);

					vec3 atmos = Atmosphere(worldDir, worldSunVector, 1.0, cloudDistance * 7e-4) * 0.4;
					//atmos += DoNightEye(Atmosphere(worldDir, -worldSunVector, 1.0, cloudDist * 2e-4) * MoonFlux);
					vec4 cloudsTemp = vec4(0.0, 0.0, 0.0, 1.0);

					// #ifdef CLOUDS_WEATHER
					// 	vec2 weatherMap = texelFetch(noisetex, ivec2(worldDay) % noiseTextureResolution, 0).yz;
					// #endif

					#ifdef CIRROCUMULUS_CLOUDS
						#ifdef CLOUDS_WEATHER
							if (cloudDynamicCovA < 0.4)
						#endif
						{
							vec4 sampleTemp = PlanarSample1(cloudDistance, cameraPosition.xz + cloudPos.xz, LdotV, 0.5, phases, worldDir);

							if (sampleTemp.a > minTransmittance) {
								float atmosFade = expf(-cloudDistance * fma(0.02, wetness, 0.12) * rcp(float(CLOUD_PLANE_ALTITUDE)));
								sampleTemp.rgb += atmos * sampleTemp.a;
								sampleTemp.rgb *= atmosFade;
								sampleTemp.rgb += skyColor * sampleTemp.a * oneMinus(atmosFade);
								cloudsTemp.rgb = sampleTemp.rgb;
								cloudsTemp.a -= sampleTemp.a;
							}
						}
					#endif
					#if CIRRUS_CLOUDS > 0
						#ifdef CLOUDS_WEATHER
							if (cloudDynamicCovB < 0.5)
						#endif
						{
							vec4 sampleTemp = PlanarSample0(cloudDistance, cameraPosition.xz + cloudPos.xz, LdotV);

							if (sampleTemp.a > minTransmittance) {
								float atmosFade = expf(-cloudDistance * fma(0.02, wetness, 0.12) * rcp(float(CLOUD_PLANE_ALTITUDE)));
								sampleTemp.rgb += atmos * sampleTemp.a;
								sampleTemp.rgb *= atmosFade;
								sampleTemp.rgb += skyColor * sampleTemp.a * oneMinus(atmosFade);
								cloudsTemp.rgb += sampleTemp.rgb * cloudsTemp.a;
								cloudsTemp.a *= 1.0 - sampleTemp.a;
							}
						}
					#endif

					if (eyeAltitude < CLOUD_PLANE_ALTITUDE) {
						cloudsData.rgb += cloudsTemp.rgb * cloudsData.a;
					} else {
						cloudsData.rgb = cloudsData.rgb * cloudsTemp.a + cloudsTemp.rgb;
					}

					cloudsData.a *= cloudsTemp.a;
				}
			}
		}
	#endif

	vec4 skyboxData;
	skyboxData.rgb = skyColor * cloudsData.a + cloudsData.rgb;

	// #ifdef AURORA
	// 	if (auroraAmount > 1e-2) skyboxData.rgb += NightAurora(worldDir) * cloudsData.a;
	// #endif

	skyboxData.a = remap(minTransmittance, 1.0, cloudsData.a);

	return clamp16F(skyboxData);
}

/////////////////////////MAIN///////////////////////////////////////////////////////////////////////////////////////////
/////////////////////////MAIN///////////////////////////////////////////////////////////////////////////////////////////
void main() {
	ivec2 texel = ivec2(gl_FragCoord.xy);

	ivec2 cloudTexel = texel * TEMPORAL_UPSCALING + checkerboardOffset[frameCounter % cloudsRenderFactor];
	vec2 coord = cloudTexel * screenPixelSize;

	CloudProperties cloudProperties = GetGlobalCloudProperties();

	// Current skybox - 当前天空盒
	if (depthMax4x4(coord) >= 1.0) {
		vec3 viewPos  = ScreenToViewSpaceRaw(vec3(coord, 1.0));
		vec3 worldDir = mat3(gbufferModelViewInverse) * normalize(viewPos);

		//float dither = texelFetch(noisetex, cloudTexel & 255, 0).a;
		float dither = R1(frameCounter / cloudsRenderFactor, texelFetch(noisetex, cloudTexel & 255, 0).a);
		// float lightNoise = InterleavedGradientNoiseCheckerboard(cloudTexel);

		skyboxData = RenderSkybox(worldDir, dither/* , lightNoise */, cloudProperties);
	}

    if (texel == ivec2(screenSize * 0.2 + 1)) colortex5Out.rgb = colorSunlight;
    if (texel == ivec2(screenSize * 0.2 + 2)) colortex5Out.rgb = colorSkylight;

	//float tileSize = floor(min(viewWidth * rcp(3.0), viewHeight * 0.5)) * 0.25;

	if (any(greaterThanEqual(texel, ceil(screenSize * 0.2)))) return;

	// Skybox reflection - 反射天空盒
	vec3 worldDir = UnprojectSky(texel * screenPixelSize);
	colortex5Out = RenderSkybox(worldDir, cloudProperties);
}
