
#include "/lib/Surface/BRDF.glsl"

const int shadowMapResolution = 2048;  // Shadowmap resolution [1024 2048 4096 8192 16384 32768]
const float	shadowDistance	  = 192.0; // [64.0 80.0 96.0 112.0 128.0 160.0 192.0 224.0 256.0 320.0 384.0 512.0 768.0 1024.0]

const float realShadowMapRes = shadowMapResolution * MC_SHADOW_QUALITY;

vec3 WorldPosToShadowProjPosBias(in vec3 worldPos, in vec3 worldNormal, out float dist, out float distortFactor) {
	vec3 sn = normalize((shadowModelView * vec4(worldNormal.xyz, 0.0)).xyz);
	sn.z = -sn.z;

	vec4 sp = shadowModelView * vec4(worldPos, 1.0);
	sp = shadowProjection * sp;
	sp /= sp.w;

	dist = length(sp.xy);
	distortFactor = oneMinus(SHADOW_MAP_BIAS) + dist * SHADOW_MAP_BIAS;

	sp.xyz += sn * 0.002 * distortFactor;
	sp.xy *= 0.95f / distortFactor;
	sp.z = mix(sp.z, 0.5, 0.8);

	return sp.xyz * 0.5f + 0.5f;
}

/*
vec3 GetWavesNormalForLight(in vec3 position) {

	vec2 coord = position.xz;
	vec3 lightVector = refract(worldLightVector, vec3(0.0, 1.0, 0.0), 1.0 / WATER_REFRACT_IOR);
	coord.x += position.y * lightVector.x / lightVector.y;
	coord.y += position.y * lightVector.z / lightVector.y;

	coord *= 0.02;
	coord = mod(coord, vec2(1.0));

	vec3 normal = DecodeNormal(texture(colortex0, coord).xy);

	return normal;
}

float CalculateWaterCaustics(in vec3 worldPos)
{
	worldPos.xyz += cameraPosition.xyz;

	vec2 dither = rand(texcoord + sin(frameTimeCounter)).xy / 4.0;

	vec3 lookupCenter = worldPos.xyz + vec3(0.0, 1.0, 0.0);

	float caustics = 0.0;

	for (float i = -1.0; i <= 1.0; i++) {
		for (float j = -1.0; j <= 1.0; j++) {
			vec2 offset = (dither + vec2(i, j)) * 0.1;
			vec3 lookupPoint = lookupCenter;
            lookupPoint.xz += offset;

			vec3 wavesNormal = GetWavesNormalForLight(lookupPoint).xzy;

			vec3 refractVector = refract(vec3(0.0, 1.0, 0.0), wavesNormal.xyz, 1.0);
			vec3 collisionPoint = lookupPoint - refractVector / refractVector.y;
            collisionPoint -= worldPos.xyz;

			//float dist = distance(collisionPoint, worldPos.xyz);
			float dist = dot(collisionPoint, collisionPoint) * 7.1;

			caustics += 1.0 - saturate(dist * 10.0);

		}
	}

	caustics = max(caustics * 0.35, 0.2) + 0.35;

	return caustics;
}
*/
vec3 VariablePenumbraShadow(in vec3 worldPos, in vec3 shadowProjPos, in float dither, in float scale, in float avgDepth) {
	const uint steps = 16u;
	const float rSteps = 1.0 / steps;

	vec3 result = vec3(0.0);

	for (uint i = 0u; i < steps; ++i) {
		float fi = float(i) + dither;
		vec2 coordOffset = cossin(fi * 10.166407) * scale * sqrt(fi * rSteps);
		vec2 sampleCoord = shadowProjPos.xy + coordOffset;

		float sampleDepth1 = textureLod(shadowtex1, vec3(sampleCoord, shadowProjPos.z), 0).x;

	#ifdef COLORED_SHADOWS
		ivec2 sampleTexel = ivec2(sampleCoord * realShadowMapRes);
		float sampleDepth0 = step(shadowProjPos.z, texelFetch(shadowtex0, sampleTexel, 0).x);
		if (sampleDepth0 != sampleDepth1) {
			vec3 albedo = pow4(texelFetch(shadowcolor0, sampleTexel, 0).rgb) * sampleDepth1;

			#ifdef WATER_CAUSTICS
				float shadowDepth = texelFetch(shadowcolor1, sampleTexel, 0).w;
				if (shadowDepth > 0.1) {
					float waterDepth = abs(shadowDepth * 512.0 - 128.0 - worldPos.y - eyeAltitude);

					float caustics = GetCaustics(worldPos, waterDepth);			
					albedo *= caustics + 0.04;
				}
			#endif

			result += albedo;
		} else 
	#endif
		{ result += sampleDepth1; }
	}

	return result * rSteps;
}

float ScreenSpaceShadow(in vec3 origin, in vec3 normal, in float noise, in float sssAmount) {
	//if (mask.hand) return 1.0;

	float fov = atan(rcp(gbufferProjection[1][1])) * 360.0 * rPI;

	vec3 lightVector = mat3(gbufferModelView) * worldLightVector;
	vec3 rayDir = lightVector * -origin.z * 0.000035 * fov;

	float NdotL = saturate(dot(lightVector, normal));

	float zThickness = 0.025 * -origin.z;
	float shadow = 1.0;
	float absorption = 0.0;
	//absorption += 0.5 * mask.grass;
	//absorption += 0.85 * mask.leaves;

	vec3 rayPos = origin;
	if (sssAmount > 1e-4) {
		absorption += saturate(sssAmount * 5.0);
	}else{
		rayPos += normal * 3e-4 * max(abs(origin.z), 0.1) / (NdotL + 0.01);
		rayPos += normal * 0.0000015 * -origin.z * fov;
		rayPos += rayDir * 1950.0 * minOf(screenPixelSize);
	}

	absorption = pow(absorption, sqrt2(length(origin)) * 0.5);

	float ds = 1.0;
	for (uint i = 0u; i < 12u; ++i) {
		rayPos += rayDir * ds;
		ds += 0.3;

		vec3 thisRayPos = rayPos + rayDir * noise * ds;
		vec2 rayProjPos = ViewToScreenSpaceRaw(thisRayPos).xy;

		if (maxOf(abs(rayProjPos)) > 1.0) break;

		#ifdef TAA_ENABLED
			rayProjPos.xy += taaOffset * 0.5;
		#endif

		float sampleDepth = -ScreenToViewSpace(GetDepth(rayProjPos.xy)); // Don't use GetDepthT()
		float depthDiff = sampleDepth - thisRayPos.z;

		if (depthDiff > 0.0 && depthDiff < zThickness) shadow *= absorption;

		if (shadow < 1e-3) break;
	}

	return shadow;
}

float CalculateFakeBouncedLight(in vec3 normal) {
	// float bounce = exp2(-max0(++normal.y)) + 0.35;

	// return saturate(bounce) * 3e-2/*  * saturate(worldLightVector.y + 0.3) */;

	normal.y = -normal.y;
	vec3 bounceVector = normalize(worldLightVector + vec3(0.0, 1.0, 0.0));
	float bounce = saturate(dot(normal, bounceVector) * 0.5 + 0.5);

	return bounce * (2.0 - bounce) * 2e-2;
}

vec3 CalculateSubsurfaceScattering(in vec3 albedo, in float sssAmount, in float sssDepth, in float LdotV) {
	//if (sssAmount < 1e-4) return vec3(0.0);

	vec3 coeff = albedo * inversesqrt(GetLuminance(albedo) + 1e-6);
	coeff = oneMinus(0.75 * saturate(coeff)) * (28.0 / sssAmount);

	vec3 subsurfaceScattering =  expf(3.0 * coeff * sssDepth) * HenyeyGreensteinPhase(-LdotV, 0.6);
		 subsurfaceScattering += expf(coeff * sssDepth) * (0.33 * HenyeyGreensteinPhase(-LdotV, 0.35) + 0.17 * rPI);

	//vec3 subsurfaceScattering = expf(coeff * sssDepth);
	//subsurfaceScattering *= HenyeyGreensteinPhase(-LdotV, 0.65) + 0.25;
	return subsurfaceScattering * sssAmount * PI;
}
